<?php
/**
 * @link http://www.yiiframework.com/
 * @copyright Copyright (c) 2008 Yii Software LLC
 * @license http://www.yiiframework.com/license/
 */

namespace yii\caching;

use Yii;

/**
 * FileCache implements a cache component using files.
 *
 * For each data value being cached, FileCache will store it in a separate file.
 * The cache files are placed under [[cachePath]]. FileCache will perform garbage collection
 * automatically to remove expired cache files.
 *
 * Please refer to [[Cache]] for common cache operations that are supported by FileCache.
 *
 * @author Qiang Xue <qiang.xue@gmail.com>
 * @since 2.0
 */
class FileCache extends Cache
{
	/**
	 * @var string the directory to store cache files. You may use path alias here.
	 */
	public $cachePath = '@app/runtime/cache';
	/**
	 * @var string cache file suffix. Defaults to '.bin'.
	 */
	public $cacheFileSuffix = '.bin';
	/**
	 * @var integer the level of sub-directories to store cache files. Defaults to 1.
	 * If the system has huge number of cache files (e.g. one million), you may use a bigger value
	 * (usually no bigger than 3). Using sub-directories is mainly to ensure the file system
	 * is not over burdened with a single directory having too many files.
	 */
	public $directoryLevel = 1;
	/**
	 * @var integer the probability (parts per million) that garbage collection (GC) should be performed
	 * when storing a piece of data in the cache. Defaults to 10, meaning 0.001% chance.
	 * This number should be between 0 and 1000000. A value 0 means no GC will be performed at all.
	 **/
	public $gcProbability = 10;

	/**
	 * Initializes this component by ensuring the existence of the cache path.
	 */
	public function init()
	{
		parent::init();
		$this->cachePath = Yii::getAlias($this->cachePath);
		if (!is_dir($this->cachePath)) {
			mkdir($this->cachePath, 0777, true);
		}
	}

	/**
	 * Retrieves a value from cache with a specified key.
	 * This is the implementation of the method declared in the parent class.
	 * @param string $key a unique key identifying the cached value
	 * @return string|boolean the value stored in cache, false if the value is not in the cache or expired.
	 */
	protected function getValue($key)
	{
		$cacheFile = $this->getCacheFile($key);
		if (($time = @filemtime($cacheFile)) > time()) {
			return @file_get_contents($cacheFile);
		} else {
			return false;
		}
	}

	/**
	 * Stores a value identified by a key in cache.
	 * This is the implementation of the method declared in the parent class.
	 *
	 * @param string $key the key identifying the value to be cached
	 * @param string $value the value to be cached
	 * @param integer $expire the number of seconds in which the cached value will expire. 0 means never expire.
	 * @return boolean true if the value is successfully stored into cache, false otherwise
	 */
	protected function setValue($key, $value, $expire)
	{
		if ($expire <= 0) {
			$expire = 31536000; // 1 year
		}
		$expire += time();

		$cacheFile = $this->getCacheFile($key);
		if ($this->directoryLevel > 0) {
			@mkdir(dirname($cacheFile), 0777, true);
		}
		if (@file_put_contents($cacheFile, $value, LOCK_EX) !== false) {
			@chmod($cacheFile, 0777);
			return @touch($cacheFile, $expire);
		} else {
			return false;
		}
	}

	/**
	 * Stores a value identified by a key into cache if the cache does not contain this key.
	 * This is the implementation of the method declared in the parent class.
	 *
	 * @param string $key the key identifying the value to be cached
	 * @param string $value the value to be cached
	 * @param integer $expire the number of seconds in which the cached value will expire. 0 means never expire.
	 * @return boolean true if the value is successfully stored into cache, false otherwise
	 */
	protected function addValue($key, $value, $expire)
	{
		$cacheFile = $this->getCacheFile($key);
		if (@filemtime($cacheFile) > time()) {
			return false;
		}
		return $this->setValue($key, $value, $expire);
	}

	/**
	 * Deletes a value with the specified key from cache
	 * This is the implementation of the method declared in the parent class.
	 * @param string $key the key of the value to be deleted
	 * @return boolean if no error happens during deletion
	 */
	protected function deleteValue($key)
	{
		$cacheFile = $this->getCacheFile($key);
		return @unlink($cacheFile);
	}

	/**
	 * Returns the cache file path given the cache key.
	 * @param string $key cache key
	 * @return string the cache file path
	 */
	protected function getCacheFile($key)
	{
		if ($this->directoryLevel > 0) {
			$base = $this->cachePath;
			for ($i = 0; $i < $this->directoryLevel; ++$i) {
				if (($prefix = substr($key, $i + $i, 2)) !== false) {
					$base .= DIRECTORY_SEPARATOR . $prefix;
				}
			}
			return $base . DIRECTORY_SEPARATOR . $key . $this->cacheFileSuffix;
		} else {
			return $this->cachePath . DIRECTORY_SEPARATOR . $key . $this->cacheFileSuffix;
		}
	}

	/**
	 * Deletes all values from cache.
	 * This is the implementation of the method declared in the parent class.
	 * @return boolean whether the flush operation was successful.
	 */
	protected function flushValues()
	{
		$this->gc(true, false);
		return true;
	}

	/**
	 * Removes expired cache files.
	 * @param boolean $force whether to enforce the garbage collection regardless of [[gcProbability]].
	 * Defaults to false, meaning the actual deletion happens with the probability as specified by [[gcProbability]].
	 * @param boolean $expiredOnly whether to removed expired cache files only.
	 * If true, all cache files under [[cachePath]] will be removed.
	 */
	public function gc($force = false, $expiredOnly = true)
	{
		if ($force || mt_rand(0, 1000000) < $this->gcProbability) {
			$this->gcRecursive($this->cachePath, $expiredOnly);
		}
	}

	/**
	 * Recursively removing expired cache files under a directory.
	 * This method is mainly used by [[gc()]].
	 * @param string $path the directory under which expired cache files are removed.
	 * @param boolean $expiredOnly whether to only remove expired cache files. If false, all files
	 * under `$path` will be removed.
	 */
	protected function gcRecursive($path, $expiredOnly)
	{
		if (($handle = opendir($path)) !== false) {
			while (($file = readdir($handle)) !== false) {
				if ($file[0] === '.') {
					continue;
				}
				$fullPath = $path . DIRECTORY_SEPARATOR . $file;
				if (is_dir($fullPath)) {
					$this->gcRecursive($fullPath, $expiredOnly);
					if (!$expiredOnly) {
						@rmdir($fullPath);
					}
				} elseif (!$expiredOnly || $expiredOnly && @filemtime($fullPath) < time()) {
					@unlink($fullPath);
				}
			}
			closedir($handle);
		}
	}
}
