<?php
/**
 * @link http://www.yiiframework.com/
 * @copyright Copyright (c) 2008 Yii Software LLC
 * @license http://www.yiiframework.com/license/
 */

namespace yii\web;

use Yii;
use yii\base\InvalidConfigException;
use yii\base\InvalidParamException;
use yii\helpers\FileHelper;
use yii\helpers\Html;
use yii\helpers\Json;
use yii\helpers\Security;
use yii\helpers\StringHelper;

/**
 * The web Response class represents an HTTP response
 *
 * It holds the [[headers]], [[cookies]] and [[content]] that is to be sent to the client.
 * It also controls the HTTP [[statusCode|status code]].
 *
 * Response is configured as an application component in [[yii\web\Application]] by default.
 * You can access that instance via `Yii::$app->response`.
 *
 * You can modify its configuration by adding an array to your application config under `components`
 * as it is shown in the following example:
 *
 * ~~~
 * 'response' => [
 *     'format' => yii\web\Response::FORMAT_JSON,
 *     'charset' => 'UTF-8',
 *     // ...
 * ]
 * ~~~
 *
 * @property CookieCollection $cookies The cookie collection. This property is read-only.
 * @property HeaderCollection $headers The header collection. This property is read-only.
 * @property boolean $isClientError Whether this response indicates a client error. This property is
 * read-only.
 * @property boolean $isEmpty Whether this response is empty. This property is read-only.
 * @property boolean $isForbidden Whether this response indicates the current request is forbidden. This
 * property is read-only.
 * @property boolean $isInformational Whether this response is informational. This property is read-only.
 * @property boolean $isInvalid Whether this response has a valid [[statusCode]]. This property is read-only.
 * @property boolean $isNotFound Whether this response indicates the currently requested resource is not
 * found. This property is read-only.
 * @property boolean $isOk Whether this response is OK. This property is read-only.
 * @property boolean $isRedirection Whether this response is a redirection. This property is read-only.
 * @property boolean $isServerError Whether this response indicates a server error. This property is
 * read-only.
 * @property boolean $isSuccessful Whether this response is successful. This property is read-only.
 * @property integer $statusCode The HTTP status code to send with the response.
 *
 * @author Qiang Xue <qiang.xue@gmail.com>
 * @author Carsten Brandt <mail@cebe.cc>
 * @since 2.0
 */
class Response extends \yii\base\Response
{
	/**
	 * @event ResponseEvent an event that is triggered at the beginning of [[send()]].
	 */
	const EVENT_BEFORE_SEND = 'beforeSend';
	/**
	 * @event ResponseEvent an event that is triggered at the end of [[send()]].
	 */
	const EVENT_AFTER_SEND = 'afterSend';
	/**
	 * @event ResponseEvent an event that is triggered right after [[prepare()]] is called in [[send()]].
	 * You may respond to this event to filter the response content before it is sent to the client.
	 */
	const EVENT_AFTER_PREPARE = 'afterPrepare';

	const FORMAT_RAW = 'raw';
	const FORMAT_HTML = 'html';
	const FORMAT_JSON = 'json';
	const FORMAT_JSONP = 'jsonp';
	const FORMAT_XML = 'xml';

	/**
	 * @var string the response format. This determines how to convert [[data]] into [[content]]
	 * when the latter is not set. By default, the following formats are supported:
	 *
	 * - [[FORMAT_RAW]]: the data will be treated as the response content without any conversion.
	 *   No extra HTTP header will be added.
	 * - [[FORMAT_HTML]]: the data will be treated as the response content without any conversion.
	 *   The "Content-Type" header will set as "text/html" if it is not set previously.
	 * - [[FORMAT_JSON]]: the data will be converted into JSON format, and the "Content-Type"
	 *   header will be set as "application/json".
	 * - [[FORMAT_JSONP]]: the data will be converted into JSONP format, and the "Content-Type"
	 *   header will be set as "text/javascript". Note that in this case `$data` must be an array
	 *   with "data" and "callback" elements. The former refers to the actual data to be sent,
	 *   while the latter refers to the name of the JavaScript callback.
	 * - [[FORMAT_XML]]: the data will be converted into XML format. Please refer to [[XmlResponseFormatter]]
	 *   for more details.
	 *
	 * You may customize the formatting process or support additional formats by configuring [[formatters]].
	 * @see formatters
	 */
	public $format = self::FORMAT_HTML;
	/**
	 * @var array the formatters for converting data into the response content of the specified [[format]].
	 * The array keys are the format names, and the array values are the corresponding configurations
	 * for creating the formatter objects.
	 * @see format
	 */
	public $formatters;
	/**
	 * @var mixed the original response data. When this is not null, it will be converted into [[content]]
	 * according to [[format]] when the response is being sent out.
	 * @see content
	 */
	public $data;
	/**
	 * @var string the response content. When [[data]] is not null, it will be converted into [[content]]
	 * according to [[format]] when the response is being sent out.
	 * @see data
	 */
	public $content;
	/**
	 * @var resource|array the stream to be sent. This can be a stream handle or an array of stream handle,
	 * the begin position and the end position. Note that when this property is set, the [[data]] and [[content]]
	 * properties will be ignored by [[send()]].
	 */
	public $stream;
	/**
	 * @var string the charset of the text response. If not set, it will use
	 * the value of [[Application::charset]].
	 */
	public $charset;
	/**
	 * @var string the HTTP status description that comes together with the status code.
	 * @see httpStatuses
	 */
	public $statusText = 'OK';
	/**
	 * @var string the version of the HTTP protocol to use. If not set, it will be determined via `$_SERVER['SERVER_PROTOCOL']`,
	 * or '1.1' if that is not available.
	 */
	public $version;
	/**
	 * @var boolean whether the response has been sent. If this is true, calling [[send()]] will do nothing.
	 */
	public $isSent = false;
	/**
	 * @var array list of HTTP status codes and the corresponding texts
	 */
	public static $httpStatuses = [
		100 => 'Continue',
		101 => 'Switching Protocols',
		102 => 'Processing',
		118 => 'Connection timed out',
		200 => 'OK',
		201 => 'Created',
		202 => 'Accepted',
		203 => 'Non-Authoritative',
		204 => 'No Content',
		205 => 'Reset Content',
		206 => 'Partial Content',
		207 => 'Multi-Status',
		208 => 'Already Reported',
		210 => 'Content Different',
		226 => 'IM Used',
		300 => 'Multiple Choices',
		301 => 'Moved Permanently',
		302 => 'Found',
		303 => 'See Other',
		304 => 'Not Modified',
		305 => 'Use Proxy',
		306 => 'Reserved',
		307 => 'Temporary Redirect',
		308 => 'Permanent Redirect',
		310 => 'Too many Redirect',
		400 => 'Bad Request',
		401 => 'Unauthorized',
		402 => 'Payment Required',
		403 => 'Forbidden',
		404 => 'Not Found',
		405 => 'Method Not Allowed',
		406 => 'Not Acceptable',
		407 => 'Proxy Authentication Required',
		408 => 'Request Time-out',
		409 => 'Conflict',
		410 => 'Gone',
		411 => 'Length Required',
		412 => 'Precondition Failed',
		413 => 'Request Entity Too Large',
		414 => 'Request-URI Too Long',
		415 => 'Unsupported Media Type',
		416 => 'Requested range unsatisfiable',
		417 => 'Expectation failed',
		418 => 'I\'m a teapot',
		422 => 'Unprocessable entity',
		423 => 'Locked',
		424 => 'Method failure',
		425 => 'Unordered Collection',
		426 => 'Upgrade Required',
		428 => 'Precondition Required',
		429 => 'Too Many Requests',
		431 => 'Request Header Fields Too Large',
		449 => 'Retry With',
		450 => 'Blocked by Windows Parental Controls',
		500 => 'Internal Server Error',
		501 => 'Not Implemented',
		502 => 'Bad Gateway ou Proxy Error',
		503 => 'Service Unavailable',
		504 => 'Gateway Time-out',
		505 => 'HTTP Version not supported',
		507 => 'Insufficient storage',
		508 => 'Loop Detected',
		509 => 'Bandwidth Limit Exceeded',
		510 => 'Not Extended',
		511 => 'Network Authentication Required',
	];

	/**
	 * @var integer the HTTP status code to send with the response.
	 */
	private $_statusCode = 200;
	/**
	 * @var HeaderCollection
	 */
	private $_headers;

	/**
	 * Initializes this component.
	 */
	public function init()
	{
		if ($this->version === null) {
			if (isset($_SERVER['SERVER_PROTOCOL']) && $_SERVER['SERVER_PROTOCOL'] === '1.0') {
				$this->version = '1.0';
			} else {
				$this->version = '1.1';
			}
		}
		if ($this->charset === null) {
			$this->charset = Yii::$app->charset;
		}
	}

	/**
	 * @return integer the HTTP status code to send with the response.
	 */
	public function getStatusCode()
	{
		return $this->_statusCode;
	}

	/**
	 * Sets the response status code.
	 * This method will set the corresponding status text if `$text` is null.
	 * @param integer $value the status code
	 * @param string $text the status text. If not set, it will be set automatically based on the status code.
	 * @throws InvalidParamException if the status code is invalid.
	 */
	public function setStatusCode($value, $text = null)
	{
		if ($value === null) {
			$value = 200;
		}
		$this->_statusCode = (int)$value;
		if ($this->getIsInvalid()) {
			throw new InvalidParamException("The HTTP status code is invalid: $value");
		}
		if ($text === null) {
			$this->statusText = isset(static::$httpStatuses[$this->_statusCode]) ? static::$httpStatuses[$this->_statusCode] : '';
		} else {
			$this->statusText = $text;
		}
	}

	/**
	 * Returns the header collection.
	 * The header collection contains the currently registered HTTP headers.
	 * @return HeaderCollection the header collection
	 */
	public function getHeaders()
	{
		if ($this->_headers === null) {
			$this->_headers = new HeaderCollection;
		}
		return $this->_headers;
	}

	/**
	 * Sends the response to the client.
	 */
	public function send()
	{
		if ($this->isSent) {
			return;
		} else {
			$this->isSent = true;
		}
		$this->trigger(self::EVENT_BEFORE_SEND);
		$this->prepare();
		$this->trigger(self::EVENT_AFTER_PREPARE);
		$this->sendHeaders();
		$this->sendContent();
		$this->trigger(self::EVENT_AFTER_SEND);
	}

	/**
	 * Clears the headers, cookies, content, status code of the response.
	 */
	public function clear()
	{
		$this->_headers = null;
		$this->_cookies = null;
		$this->_statusCode = 200;
		$this->statusText = 'OK';
		$this->data = null;
		$this->stream = null;
		$this->content = null;
		$this->isSent = false;
	}

	/**
	 * Sends the response headers to the client
	 */
	protected function sendHeaders()
	{
		if (headers_sent()) {
			return;
		}
		$statusCode = $this->getStatusCode();
		header("HTTP/{$this->version} $statusCode {$this->statusText}");
		if ($this->_headers) {
			$headers = $this->getHeaders();
			foreach ($headers as $name => $values) {
				$name = str_replace(' ', '-', ucwords(str_replace('-', ' ', $name)));
				foreach ($values as $value) {
					header("$name: $value", false);
				}
			}
		}
		$this->sendCookies();
	}

	/**
	 * Sends the cookies to the client.
	 */
	protected function sendCookies()
	{
		if ($this->_cookies === null) {
			return;
		}
		$request = Yii::$app->getRequest();
		if ($request->enableCookieValidation) {
			$validationKey = $request->getCookieValidationKey();
		}
		foreach ($this->getCookies() as $cookie) {
			$value = $cookie->value;
			if ($cookie->expire != 1  && isset($validationKey)) {
				$value = Security::hashData(serialize($value), $validationKey);
			}
			setcookie($cookie->name, $value, $cookie->expire, $cookie->path, $cookie->domain, $cookie->secure, $cookie->httpOnly);
		}
		$this->getCookies()->removeAll();
	}

	/**
	 * Sends the response content to the client
	 */
	protected function sendContent()
	{
		if ($this->stream === null) {
			echo $this->content;
			return;
		}

		set_time_limit(0); // Reset time limit for big files
		$chunkSize = 8 * 1024 * 1024; // 8MB per chunk

		if (is_array($this->stream)) {
			list ($handle, $begin, $end) = $this->stream;
			fseek($handle, $begin);
			while (!feof($handle) && ($pos = ftell($handle)) <= $end) {
				if ($pos + $chunkSize > $end) {
					$chunkSize = $end - $pos + 1;
				}
				echo fread($handle, $chunkSize);
				flush(); // Free up memory. Otherwise large files will trigger PHP's memory limit.
			}
			fclose($handle);
		} else {
			while (!feof($this->stream)) {
				echo fread($this->stream, $chunkSize);
				flush();
			}
			fclose($this->stream);
		}
	}

	/**
	 * Sends a file to the browser.
	 *
	 * Note that this method only prepares the response for file sending. The file is not sent
	 * until [[send()]] is called explicitly or implicitly. The latter is done after you return from a controller action.
	 *
	 * @param string $filePath the path of the file to be sent.
	 * @param string $attachmentName the file name shown to the user. If null, it will be determined from `$filePath`.
	 * @param string $mimeType the MIME type of the content. If null, it will be guessed based on `$filePath`
	 * @return static the response object itself
	 */
	public function sendFile($filePath, $attachmentName = null, $mimeType = null)
	{
		if ($mimeType === null && ($mimeType = FileHelper::getMimeTypeByExtension($filePath)) === null) {
			$mimeType = 'application/octet-stream';
		}
		if ($attachmentName === null) {
			$attachmentName = basename($filePath);
		}
		$handle = fopen($filePath, 'rb');
		$this->sendStreamAsFile($handle, $attachmentName, $mimeType);

		return $this;
	}

	/**
	 * Sends the specified content as a file to the browser.
	 *
	 * Note that this method only prepares the response for file sending. The file is not sent
	 * until [[send()]] is called explicitly or implicitly. The latter is done after you return from a controller action.
	 *
	 * @param string $content the content to be sent. The existing [[content]] will be discarded.
	 * @param string $attachmentName the file name shown to the user.
	 * @param string $mimeType the MIME type of the content.
	 * @return static the response object itself
	 * @throws HttpException if the requested range is not satisfiable
	 */
	public function sendContentAsFile($content, $attachmentName, $mimeType = 'application/octet-stream')
	{
		$headers = $this->getHeaders();
		$contentLength = StringHelper::byteLength($content);
		$range = $this->getHttpRange($contentLength);
		if ($range === false) {
			$headers->set('Content-Range', "bytes */$contentLength");
			throw new HttpException(416, 'Requested range not satisfiable');
		}

		$headers->setDefault('Pragma', 'public')
			->setDefault('Accept-Ranges', 'bytes')
			->setDefault('Expires', '0')
			->setDefault('Content-Type', $mimeType)
			->setDefault('Cache-Control', 'must-revalidate, post-check=0, pre-check=0')
			->setDefault('Content-Transfer-Encoding', 'binary')
			->setDefault('Content-Length', StringHelper::byteLength($content))
			->setDefault('Content-Disposition', "attachment; filename=\"$attachmentName\"");

		list($begin, $end) = $range;
		if ($begin !=0 || $end != $contentLength - 1) {
			$this->setStatusCode(206);
			$headers->set('Content-Range', "bytes $begin-$end/$contentLength");
			$this->content = StringHelper::byteSubstr($content, $begin, $end - $begin + 1);
		} else {
			$this->setStatusCode(200);
			$this->content = $content;
		}

		$this->format = self::FORMAT_RAW;

		return $this;
	}

	/**
	 * Sends the specified stream as a file to the browser.
	 *
	 * Note that this method only prepares the response for file sending. The file is not sent
	 * until [[send()]] is called explicitly or implicitly. The latter is done after you return from a controller action.
	 *
	 * @param resource $handle the handle of the stream to be sent.
	 * @param string $attachmentName the file name shown to the user.
	 * @param string $mimeType the MIME type of the stream content.
	 * @return static the response object itself
	 * @throws HttpException if the requested range cannot be satisfied.
	 */
	public function sendStreamAsFile($handle, $attachmentName, $mimeType = 'application/octet-stream')
	{
		$headers = $this->getHeaders();
		fseek($handle, 0, SEEK_END);
		$fileSize = ftell($handle);

		$range = $this->getHttpRange($fileSize);
		if ($range === false) {
			$headers->set('Content-Range', "bytes */$fileSize");
			throw new HttpException(416, 'Requested range not satisfiable');
		}

		list($begin, $end) = $range;
		if ($begin !=0 || $end != $fileSize - 1) {
			$this->setStatusCode(206);
			$headers->set('Content-Range', "bytes $begin-$end/$fileSize");
		} else {
			$this->setStatusCode(200);
		}

		$length = $end - $begin + 1;

		$headers->setDefault('Pragma', 'public')
			->setDefault('Accept-Ranges', 'bytes')
			->setDefault('Expires', '0')
			->setDefault('Content-Type', $mimeType)
			->setDefault('Cache-Control', 'must-revalidate, post-check=0, pre-check=0')
			->setDefault('Content-Transfer-Encoding', 'binary')
			->setDefault('Content-Length', $length)
			->setDefault('Content-Disposition', "attachment; filename=\"$attachmentName\"");
		$this->format = self::FORMAT_RAW;
		$this->stream = [$handle, $begin, $end];

		return $this;
	}

	/**
	 * Determines the HTTP range given in the request.
	 * @param integer $fileSize the size of the file that will be used to validate the requested HTTP range.
	 * @return array|boolean the range (begin, end), or false if the range request is invalid.
	 */
	protected function getHttpRange($fileSize)
	{
		if (!isset($_SERVER['HTTP_RANGE']) || $_SERVER['HTTP_RANGE'] === '-') {
			return [0, $fileSize - 1];
		}
		if (!preg_match('/^bytes=(\d*)-(\d*)$/', $_SERVER['HTTP_RANGE'], $matches)) {
			return false;
		}
		if ($matches[1] === '') {
			$start = $fileSize - $matches[2];
			$end = $fileSize - 1;
		} elseif ($matches[2] !== '') {
			$start = $matches[1];
			$end = $matches[2];
			if ($end >= $fileSize) {
				$end = $fileSize - 1;
			}
		} else {
			$start = $matches[1];
			$end = $fileSize - 1;
		}
		if ($start < 0 || $start > $end) {
			return false;
		} else {
			return [$start, $end];
		}
	}

	/**
	 * Sends existing file to a browser as a download using x-sendfile.
	 *
	 * X-Sendfile is a feature allowing a web application to redirect the request for a file to the webserver
	 * that in turn processes the request, this way eliminating the need to perform tasks like reading the file
	 * and sending it to the user. When dealing with a lot of files (or very big files) this can lead to a great
	 * increase in performance as the web application is allowed to terminate earlier while the webserver is
	 * handling the request.
	 *
	 * The request is sent to the server through a special non-standard HTTP-header.
	 * When the web server encounters the presence of such header it will discard all output and send the file
	 * specified by that header using web server internals including all optimizations like caching-headers.
	 *
	 * As this header directive is non-standard different directives exists for different web servers applications:
	 *
	 * - Apache: [X-Sendfile](http://tn123.org/mod_xsendfile)
	 * - Lighttpd v1.4: [X-LIGHTTPD-send-file](http://redmine.lighttpd.net/projects/lighttpd/wiki/X-LIGHTTPD-send-file)
	 * - Lighttpd v1.5: [X-Sendfile](http://redmine.lighttpd.net/projects/lighttpd/wiki/X-LIGHTTPD-send-file)
	 * - Nginx: [X-Accel-Redirect](http://wiki.nginx.org/XSendfile)
	 * - Cherokee: [X-Sendfile and X-Accel-Redirect](http://www.cherokee-project.com/doc/other_goodies.html#x-sendfile)
	 *
	 * So for this method to work the X-SENDFILE option/module should be enabled by the web server and
	 * a proper xHeader should be sent.
	 *
	 * **Note**
	 *
	 * This option allows to download files that are not under web folders, and even files that are otherwise protected
	 * (deny from all) like `.htaccess`.
	 *
	 * **Side effects**
	 *
	 * If this option is disabled by the web server, when this method is called a download configuration dialog
	 * will open but the downloaded file will have 0 bytes.
	 *
	 * **Known issues**
	 *
	 * There is a Bug with Internet Explorer 6, 7 and 8 when X-SENDFILE is used over an SSL connection, it will show
	 * an error message like this: "Internet Explorer was not able to open this Internet site. The requested site
	 * is either unavailable or cannot be found.". You can work around this problem by removing the `Pragma`-header.
	 *
	 * **Example**
	 *
	 * ~~~
	 * Yii::$app->response->xSendFile('/home/user/Pictures/picture1.jpg');
	 * ~~~
	 *
	 * @param string $filePath file name with full path
	 * @param string $mimeType the MIME type of the file. If null, it will be determined based on `$filePath`.
	 * @param string $attachmentName file name shown to the user. If null, it will be determined from `$filePath`.
	 * @param string $xHeader the name of the x-sendfile header.
	 * @return static the response object itself
	 */
	public function xSendFile($filePath, $attachmentName = null, $mimeType = null, $xHeader = 'X-Sendfile')
	{
		if ($mimeType === null && ($mimeType = FileHelper::getMimeTypeByExtension($filePath)) === null) {
			$mimeType = 'application/octet-stream';
		}
		if ($attachmentName === null) {
			$attachmentName = basename($filePath);
		}

		$this->getHeaders()
			->setDefault($xHeader, $filePath)
			->setDefault('Content-Type', $mimeType)
			->setDefault('Content-Disposition', "attachment; filename=\"$attachmentName\"");

		return $this;
	}

	/**
	 * Redirects the browser to the specified URL.
	 *
	 * This method adds a "Location" header to the current response. Note that it does not send out
	 * the header until [[send()]] is called. In a controller action you may use this method as follows:
	 *
	 * ~~~
	 * return Yii::$app->getResponse()->redirect($url);
	 * ~~~
	 *
	 * In other places, if you want to send out the "Location" header immediately, you should use
	 * the following code:
	 *
	 * ~~~
	 * Yii::$app->getResponse()->redirect($url)->send();
	 * return;
	 * ~~~
	 *
	 * In AJAX mode, this normally will not work as expected unless there are some
	 * client-side JavaScript code handling the redirection. To help achieve this goal,
	 * this method will send out a "X-Redirect" header instead of "Location".
	 *
	 * If you use the "yii" JavaScript module, it will handle the AJAX redirection as
	 * described above. Otherwise, you should write the following JavaScript code to
	 * handle the redirection:
	 *
	 * ~~~
	 * $document.ajaxComplete(function (event, xhr, settings) {
	 *     var url = xhr.getResponseHeader('X-Redirect');
	 *     if (url) {
	 *         window.location = url;
	 *     }
	 * });
	 * ~~~
	 *
	 * @param string|array $url the URL to be redirected to. This can be in one of the following formats:
	 *
	 * - a string representing a URL (e.g. "http://example.com")
	 * - a string representing a URL alias (e.g. "@example.com")
	 * - an array in the format of `[$route, ...name-value pairs...]` (e.g. `['site/index', 'ref' => 1]`).
	 *   Note that the route is with respect to the whole application, instead of relative to a controller or module.
	 *   [[Html::url()]] will be used to convert the array into a URL.
	 *
	 * Any relative URL will be converted into an absolute one by prepending it with the host info
	 * of the current request.
	 *
	 * @param integer $statusCode the HTTP status code. Defaults to 302.
	 * See <http://www.w3.org/Protocols/rfc2616/rfc2616-sec10.html>
	 * for details about HTTP status code
	 * @return static the response object itself
	 */
	public function redirect($url, $statusCode = 302)
	{
		if (is_array($url) && isset($url[0])) {
			// ensure the route is absolute
			$url[0] = '/' . ltrim($url[0], '/');
		}
		$url = Html::url($url);
		if (strpos($url, '/') === 0 && strpos($url, '//') !== 0) {
			$url = Yii::$app->getRequest()->getHostInfo() . $url;
		}

		if (Yii::$app->getRequest()->getIsAjax()) {
			$this->getHeaders()->set('X-Redirect', $url);
		} else {
			$this->getHeaders()->set('Location', $url);
		}
		$this->setStatusCode($statusCode);

		return $this;
	}

	/**
	 * Refreshes the current page.
	 * The effect of this method call is the same as the user pressing the refresh button of his browser
	 * (without re-posting data).
	 *
	 * In a controller action you may use this method like this:
	 *
	 * ~~~
	 * return Yii::$app->getResponse()->refresh();
	 * ~~~
	 *
	 * @param string $anchor the anchor that should be appended to the redirection URL.
	 * Defaults to empty. Make sure the anchor starts with '#' if you want to specify it.
	 * @return Response the response object itself
	 */
	public function refresh($anchor = '')
	{
		return $this->redirect(Yii::$app->getRequest()->getUrl() . $anchor);
	}

	private $_cookies;

	/**
	 * Returns the cookie collection.
	 * Through the returned cookie collection, you add or remove cookies as follows,
	 *
	 * ~~~
	 * // add a cookie
	 * $response->cookies->add(new Cookie([
	 *     'name' => $name,
	 *     'value' => $value,
	 * ]);
	 *
	 * // remove a cookie
	 * $response->cookies->remove('name');
	 * // alternatively
	 * unset($response->cookies['name']);
	 * ~~~
	 *
	 * @return CookieCollection the cookie collection.
	 */
	public function getCookies()
	{
		if ($this->_cookies === null) {
			$this->_cookies = new CookieCollection;
		}
		return $this->_cookies;
	}

	/**
	 * @return boolean whether this response has a valid [[statusCode]].
	 */
	public function getIsInvalid()
	{
		return $this->getStatusCode() < 100 || $this->getStatusCode() >= 600;
	}

	/**
	 * @return boolean whether this response is informational
	 */
	public function getIsInformational()
	{
		return $this->getStatusCode() >= 100 && $this->getStatusCode() < 200;
	}

	/**
	 * @return boolean whether this response is successful
	 */
	public function getIsSuccessful()
	{
		return $this->getStatusCode() >= 200 && $this->getStatusCode() < 300;
	}

	/**
	 * @return boolean whether this response is a redirection
	 */
	public function getIsRedirection()
	{
		return $this->getStatusCode() >= 300 && $this->getStatusCode() < 400;
	}

	/**
	 * @return boolean whether this response indicates a client error
	 */
	public function getIsClientError()
	{
		return $this->getStatusCode() >= 400 && $this->getStatusCode() < 500;
	}

	/**
	 * @return boolean whether this response indicates a server error
	 */
	public function getIsServerError()
	{
		return $this->getStatusCode() >= 500 && $this->getStatusCode() < 600;
	}

	/**
	 * @return boolean whether this response is OK
	 */
	public function getIsOk()
	{
		return $this->getStatusCode() == 200;
	}

	/**
	 * @return boolean whether this response indicates the current request is forbidden
	 */
	public function getIsForbidden()
	{
		return $this->getStatusCode() == 403;
	}

	/**
	 * @return boolean whether this response indicates the currently requested resource is not found
	 */
	public function getIsNotFound()
	{
		return $this->getStatusCode() == 404;
	}

	/**
	 * @return boolean whether this response is empty
	 */
	public function getIsEmpty()
	{
		return in_array($this->getStatusCode(), [201, 204, 304]);
	}

	/**
	 * Prepares for sending the response.
	 * The default implementation will convert [[data]] into [[content]] and set headers accordingly.
	 * @throws InvalidConfigException if the formatter for the specified format is invalid or [[format]] is not supported
	 */
	protected function prepare()
	{
		if ($this->stream !== null || $this->data === null) {
			return;
		}

		if (isset($this->formatters[$this->format])) {
			$formatter = $this->formatters[$this->format];
			if (!is_object($formatter)) {
				$formatter = Yii::createObject($formatter);
			}
			if ($formatter instanceof ResponseFormatterInterface) {
				$formatter->format($this);
			} else {
				throw new InvalidConfigException("The '{$this->format}' response formatter is invalid. It must implement the ResponseFormatterInterface.");
			}
		} else {
			switch ($this->format) {
				case self::FORMAT_HTML:
					$this->getHeaders()->setDefault('Content-Type', 'text/html; charset=' . $this->charset);
					$this->content = $this->data;
					break;
				case self::FORMAT_RAW:
					$this->content = $this->data;
					break;
				case self::FORMAT_JSON:
					$this->getHeaders()->set('Content-Type', 'application/json; charset=UTF-8');
					$this->content = Json::encode($this->data);
					break;
				case self::FORMAT_JSONP:
					$this->getHeaders()->set('Content-Type', 'text/javascript; charset=' . $this->charset);
					if (is_array($this->data) && isset($this->data['data'], $this->data['callback'])) {
						$this->content = sprintf('%s(%s);', $this->data['callback'], Json::encode($this->data['data']));
					} else {
						$this->content = '';
						Yii::warning("The 'jsonp' response requires that the data be an array consisting of both 'data' and 'callback' elements.", __METHOD__);
					}
					break;
				case self::FORMAT_XML:
					Yii::createObject(XmlResponseFormatter::className())->format($this);
					break;
				default:
					throw new InvalidConfigException("Unsupported response format: {$this->format}");
			}
		}

		if (is_array($this->content)) {
			$this->content = 'array()';
		} elseif (is_object($this->content)) {
			$this->content = method_exists($this->content, '__toString') ? $this->content->__toString() : get_class($this->content);
		}
	}
}
