<?php
/**
 * @link http://www.yiiframework.com/
 * @copyright Copyright (c) 2008 Yii Software LLC
 * @license http://www.yiiframework.com/license/
 */

namespace yii\redis;

use yii\base\InvalidConfigException;
use yii\db\BaseActiveRecord;
use yii\helpers\Inflector;
use yii\helpers\StringHelper;

/**
 * ActiveRecord is the base class for classes representing relational data in terms of objects.
 *
 * This class implements the ActiveRecord pattern for the [redis](http://redis.io/) key-value store.
 *
 * For defining a record a subclass should at least implement the [[attributes()]] method to define
 * attributes. A primary key can be defined via [[primaryKey()]] which defaults to `id` if not specified.
 *
 * The following is an example model called `Customer`:
 *
 * ```php
 * class Customer extends \yii\redis\ActiveRecord
 * {
 *     public function attributes()
 *     {
 *         return ['id', 'name', 'address', 'registration_date'];
 *     }
 * }
 * ```
 *
 * @author Carsten Brandt <mail@cebe.cc>
 * @since 2.0
 */
class ActiveRecord extends BaseActiveRecord
{
	/**
	 * Returns the database connection used by this AR class.
	 * By default, the "redis" application component is used as the database connection.
	 * You may override this method if you want to use a different database connection.
	 * @return Connection the database connection used by this AR class.
	 */
	public static function getDb()
	{
		return \Yii::$app->getComponent('redis');
	}

	/**
	 * Creates an [[ActiveQuery]] instance.
	 *
	 * This method is called by [[find()]], [[findBySql()]] to start a SELECT query but also
	 * by [[hasOne()]] and [[hasMany()]] to create a relational query.
	 * You may override this method to return a customized query (e.g. `CustomerQuery` specified
	 * written for querying `Customer` purpose.)
	 *
	 * You may also define default conditions that should apply to all queries unless overridden:
	 *
	 * ```php
	 * public static function createQuery($config = [])
	 * {
	 *     return parent::createQuery($config)->where(['deleted' => false]);
	 * }
	 * ```
	 *
	 * Note that all queries should use [[Query::andWhere()]] and [[Query::orWhere()]] to keep the
	 * default condition. Using [[Query::where()]] will override the default condition.
	 *
	 * @param array $config the configuration passed to the ActiveQuery class.
	 * @return ActiveQuery the newly created [[ActiveQuery]] instance.
	 */
	public static function createQuery($config = [])
	{
		$config['modelClass'] = get_called_class();
		return new ActiveQuery($config);
	}

	/**
	 * Returns the primary key name(s) for this AR class.
	 * This method should be overridden by child classes to define the primary key.
	 *
	 * Note that an array should be returned even when it is a single primary key.
	 *
	 * @return string[] the primary keys of this record.
	 */
	public static function primaryKey()
	{
		return ['id'];
	}

	/**
	 * Returns the list of all attribute names of the model.
	 * This method must be overridden by child classes to define available attributes.
	 * @return array list of attribute names.
	 */
	public function attributes()
	{
		throw new InvalidConfigException('The attributes() method of redis ActiveRecord has to be implemented by child classes.');
	}

	/**
	 * Declares prefix of the key that represents the keys that store this records in redis.
	 * By default this method returns the class name as the table name by calling [[Inflector::camel2id()]].
	 * For example, 'Customer' becomes 'customer', and 'OrderItem' becomes
	 * 'order_item'. You may override this method if you want different key naming.
	 * @return string the prefix to apply to all AR keys
	 */
	public static function keyPrefix()
	{
		return Inflector::camel2id(StringHelper::basename(get_called_class()), '_');
	}

	/**
	 * @inheritdoc
	 */
	public function insert($runValidation = true, $attributes = null)
	{
		if ($runValidation && !$this->validate($attributes)) {
			return false;
		}
		if ($this->beforeSave(true)) {
			$db = static::getDb();
			$values = $this->getDirtyAttributes($attributes);
			$pk = [];
//			if ($values === []) {
			foreach ($this->primaryKey() as $key) {
				$pk[$key] = $values[$key] = $this->getAttribute($key);
				if ($pk[$key] === null) {
					$pk[$key] = $values[$key] = $db->executeCommand('INCR', [static::keyPrefix() . ':s:' . $key]);
					$this->setAttribute($key, $values[$key]);
				}
			}
//			}
			// save pk in a findall pool
			$db->executeCommand('RPUSH', [static::keyPrefix(), static::buildKey($pk)]);

			$key = static::keyPrefix() . ':a:' . static::buildKey($pk);
			// save attributes
			$args = [$key];
			foreach ($values as $attribute => $value) {
				$args[] = $attribute;
				$args[] = $value;
			}
			$db->executeCommand('HMSET', $args);

			$this->setOldAttributes($values);
			$this->afterSave(true);
			return true;
		}
		return false;
	}

	/**
	 * Updates the whole table using the provided attribute values and conditions.
	 * For example, to change the status to be 1 for all customers whose status is 2:
	 *
	 * ~~~
	 * Customer::updateAll(['status' => 1], ['id' => 2]);
	 * ~~~
	 *
	 * @param array $attributes attribute values (name-value pairs) to be saved into the table
	 * @param array $condition the conditions that will be put in the WHERE part of the UPDATE SQL.
	 * Please refer to [[ActiveQuery::where()]] on how to specify this parameter.
	 * @return integer the number of rows updated
	 */
	public static function updateAll($attributes, $condition = null)
	{
		if (empty($attributes)) {
			return 0;
		}
		$db = static::getDb();
		$n = 0;
		foreach (static::fetchPks($condition) as $pk) {
			$newPk = $pk;
			$pk = static::buildKey($pk);
			$key = static::keyPrefix() . ':a:' . $pk;
			// save attributes
			$args = [$key];
			foreach ($attributes as $attribute => $value) {
				if (isset($newPk[$attribute])) {
					$newPk[$attribute] = $value;
				}
				$args[] = $attribute;
				$args[] = $value;
			}
			$newPk = static::buildKey($newPk);
			$newKey = static::keyPrefix() . ':a:' . $newPk;
			// rename index if pk changed
			if ($newPk != $pk) {
				$db->executeCommand('MULTI');
				$db->executeCommand('HMSET', $args);
				$db->executeCommand('LINSERT', [static::keyPrefix(), 'AFTER', $pk, $newPk]);
				$db->executeCommand('LREM', [static::keyPrefix(), 0, $pk]);
				$db->executeCommand('RENAME', [$key, $newKey]);
				$db->executeCommand('EXEC');
			} else {
				$db->executeCommand('HMSET', $args);
			}
			$n++;
		}
		return $n;
	}

	/**
	 * Updates the whole table using the provided counter changes and conditions.
	 * For example, to increment all customers' age by 1,
	 *
	 * ~~~
	 * Customer::updateAllCounters(['age' => 1]);
	 * ~~~
	 *
	 * @param array $counters the counters to be updated (attribute name => increment value).
	 * Use negative values if you want to decrement the counters.
	 * @param array $condition the conditions that will be put in the WHERE part of the UPDATE SQL.
	 * Please refer to [[ActiveQuery::where()]] on how to specify this parameter.
	 * @return integer the number of rows updated
	 */
	public static function updateAllCounters($counters, $condition = null)
	{
		if (empty($counters)) {
			return 0;
		}
		$db = static::getDb();
		$n = 0;
		foreach (static::fetchPks($condition) as $pk) {
			$key = static::keyPrefix() . ':a:' . static::buildKey($pk);
			foreach ($counters as $attribute => $value) {
				$db->executeCommand('HINCRBY', [$key, $attribute, $value]);
			}
			$n++;
		}
		return $n;
	}

	/**
	 * Deletes rows in the table using the provided conditions.
	 * WARNING: If you do not specify any condition, this method will delete ALL rows in the table.
	 *
	 * For example, to delete all customers whose status is 3:
	 *
	 * ~~~
	 * Customer::deleteAll(['status' => 3]);
	 * ~~~
	 *
	 * @param array $condition the conditions that will be put in the WHERE part of the DELETE SQL.
	 * Please refer to [[ActiveQuery::where()]] on how to specify this parameter.
	 * @return integer the number of rows deleted
	 */
	public static function deleteAll($condition = null)
	{
		$db = static::getDb();
		$attributeKeys = [];
		$pks = static::fetchPks($condition);
		$db->executeCommand('MULTI');
		foreach ($pks as $pk) {
			$pk = static::buildKey($pk);
			$db->executeCommand('LREM', [static::keyPrefix(), 0, $pk]);
			$attributeKeys[] = static::keyPrefix() . ':a:' . $pk;
		}
		if (empty($attributeKeys)) {
			$db->executeCommand('EXEC');
			return 0;
		}
		$db->executeCommand('DEL', $attributeKeys);
		$result = $db->executeCommand('EXEC');
		return end($result);
	}

	private static function fetchPks($condition)
	{
		$query = static::createQuery();
		$query->where($condition);
		$records = $query->asArray()->all(); // TODO limit fetched columns to pk
		$primaryKey = static::primaryKey();

		$pks = [];
		foreach ($records as $record) {
			$pk = [];
			foreach ($primaryKey as $key) {
				$pk[$key] = $record[$key];
			}
			$pks[] = $pk;
		}
		return $pks;
	}

	/**
	 * Builds a normalized key from a given primary key value.
	 *
	 * @param mixed $key the key to be normalized
	 * @return string the generated key
	 */
	public static function buildKey($key)
	{
		if (is_numeric($key)) {
			return $key;
		} elseif (is_string($key)) {
			return ctype_alnum($key) && StringHelper::byteLength($key) <= 32 ? $key : md5($key);
		} elseif (is_array($key)) {
			if (count($key) == 1) {
				return self::buildKey(reset($key));
			}
			ksort($key); // ensure order is always the same
			$isNumeric = true;
			foreach ($key as $value) {
				if (!is_numeric($value)) {
					$isNumeric = false;
				}
			}
			if ($isNumeric) {
				return implode('-', $key);
			}
		}
		return md5(json_encode($key));
	}
}
