<?php
/**
 * @link http://www.yiiframework.com/
 * @copyright Copyright (c) 2008 Yii Software LLC
 * @license http://www.yiiframework.com/license/
 */

namespace yii\db\cubrid;

use yii\base\InvalidParamException;

/**
 * QueryBuilder is the query builder for CUBRID databases (version 9.1.x and higher).
 *
 * @author Carsten Brandt <mail@cebe.cc>
 * @since 2.0
 */
class QueryBuilder extends \yii\db\QueryBuilder
{
	/**
	 * @var array mapping from abstract column types (keys) to physical column types (values).
	 */
	public $typeMap = [
		Schema::TYPE_PK => 'int NOT NULL AUTO_INCREMENT PRIMARY KEY',
		Schema::TYPE_BIGPK => 'bigint NOT NULL AUTO_INCREMENT PRIMARY KEY',
		Schema::TYPE_STRING => 'varchar(255)',
		Schema::TYPE_TEXT => 'varchar',
		Schema::TYPE_SMALLINT => 'smallint',
		Schema::TYPE_INTEGER => 'int',
		Schema::TYPE_BIGINT => 'bigint',
		Schema::TYPE_FLOAT => 'float(7)',
		Schema::TYPE_DECIMAL => 'decimal(10,0)',
		Schema::TYPE_DATETIME => 'datetime',
		Schema::TYPE_TIMESTAMP => 'timestamp',
		Schema::TYPE_TIME => 'time',
		Schema::TYPE_DATE => 'date',
		Schema::TYPE_BINARY => 'blob',
		Schema::TYPE_BOOLEAN => 'smallint',
		Schema::TYPE_MONEY => 'decimal(19,4)',
	];

	/**
	 * Creates a SQL statement for resetting the sequence value of a table's primary key.
	 * The sequence will be reset such that the primary key of the next new row inserted
	 * will have the specified value or 1.
	 * @param string $tableName the name of the table whose primary key sequence will be reset
	 * @param mixed $value the value for the primary key of the next new row inserted. If this is not set,
	 * the next new row's primary key will have a value 1.
	 * @return string the SQL statement for resetting sequence
	 * @throws InvalidParamException if the table does not exist or there is no sequence associated with the table.
	 */
	public function resetSequence($tableName, $value = null)
	{
		$table = $this->db->getTableSchema($tableName);
		if ($table !== null && $table->sequenceName !== null) {
			$tableName = $this->db->quoteTableName($tableName);
			if ($value === null) {
				$key = reset($table->primaryKey);
				$value = (int)$this->db->createCommand("SELECT MAX(`$key`) FROM " . $this->db->schema->quoteTableName($tableName))->queryScalar() + 1;
			} else {
				$value = (int)$value;
			}
			return "ALTER TABLE " . $this->db->schema->quoteTableName($tableName) . " AUTO_INCREMENT=$value;";
		} elseif ($table === null) {
			throw new InvalidParamException("Table not found: $tableName");
		} else {
			throw new InvalidParamException("There is not sequence associated with table '$tableName'.");
		}
	}

	/**
	 * {@inheritdoc}
	 */
	public function buildLimit($limit, $offset)
	{
		$sql = '';
		// limit is not optional in CUBRID
		// http://www.cubrid.org/manual/90/en/LIMIT%20Clause
		// "You can specify a very big integer for row_count to display to the last row, starting from a specific row."
		if ($limit !== null && $limit >= 0) {
			$sql = 'LIMIT ' . (int)$limit;
			if ($offset > 0) {
				$sql .= ' OFFSET ' . (int)$offset;
			}
		} elseif ($offset > 0) {
			$sql = 'LIMIT 9223372036854775807 OFFSET ' . (int)$offset; // 2^63-1
		}
		return $sql;
	}
}
