<?php
/**
 * Genesis Framework.
 *
 * WARNING: This file is part of the core Genesis Framework. DO NOT edit this file under any circumstances.
 * Please do all modifications in the form of a child theme.
 *
 * @package Genesis\Markup
 * @author  StudioPress
 * @license GPL-2.0+
 * @link    http://my.studiopress.com/themes/genesis/
 */

/**
 * Output markup conditionally.
 *
 * Supported keys for `$args` are:
 *
 *  - `html5` (`sprintf()` pattern markup),
 *  - `xhtml` (XHTML markup),
 *  - `context` (name of context),
 *  - `echo` (default is true).
 *
 * If the child theme supports HTML5, then this function will output the `html5` value, with a call to `genesis_attr()`
 * with the same context added in. Otherwise, it will output the `xhtml` value.
 *
 * Applies a `genesis_markup_{context}` filter early to allow shortcutting the function.
 *
 * Applies a `genesis_markup_{context}_output` filter at the end.
 *
 * @since 1.9.0
 *
 * @param array $args {
 *     Contains markup arguments.
 *     @type string html5   Legacy HTML5 markup.
 *     @type string xhtml   Legacy XHTML markup.
 *     @type string context Markup context.
 *     @type string open    Opening HTML markup.
 *     @type string close   Closing HTML markup.
 *     @type string content Content to be placed between open and close HTML markup.
 *     @type bool   echo    Flag indicating whether to echo or return the resultant string.
 * }
 * @return string|null Markup.
 */
function genesis_markup( $args = array() ) {

	$defaults = array(
		'html5'   => '',
		'xhtml'   => '',
		'context' => '',
		'open'    => '',
		'close'   => '',
		'content' => '',
		'echo'    => true,
		'params'  => array(),
	);

	$args = wp_parse_args( $args, $defaults );

	/**
	 * Filter to short circuit the markup API.
	 *
	 * @since 1.9.0
	 *
	 * @param bool  false Flag indicating short curcuit content.
	 * @param array $args Array with markup arguments.
	 *
	 * @see genesis_markup $args Array.
	 */
	$pre = apply_filters( "genesis_markup_{$args['context']}", false, $args );
	if ( false !== $pre ) {

		if ( ! $args['echo'] ) {
			return $pre;
		}

		echo $pre;
		return null;

	}

	if ( $args['html5'] || $args['xhtml'] ) {

		// If HTML5, return HTML5 tag. Maybe add attributes. Else XHTML.
		if ( genesis_html5() ) {
			$tag = $args['context'] ? sprintf( $args['html5'], genesis_attr( $args['context'] ) ) : $args['html5'];
		} else {
			$tag = $args['xhtml'];
		}

		/**
		 * Legacy contextual filter to modify 'xhtml' or 'html5' output markup.
		 *
		 * @since 1.9.0
		 *
		 * @param string $tag  HTML tag being processed by the API.
		 * @param array  $args Array with markup arguments.
		 *
		 * @see genesis_markup $args Array.
		 */
		$tag = $args['context'] ? apply_filters( "genesis_markup_{$args['context']}_output", $tag, $args ) : $tag;

		if ( ! $args['echo'] ) {
			return $tag;
		}

		echo $tag;
		return null;

	}

	// Add attributes to open tag.
	if ( $args['context'] ) {

		$open = $args['open'] ? sprintf( $args['open'], genesis_attr( $args['context'], array(), $args ) ) : '';

		/**
		 * Contextual filter to modify 'open' markup.
		 *
		 * @since 2.4.0
		 *
		 * @param string $open HTML tag being processed by the API.
		 * @param array  $args Array with markup arguments.
		 *
		 * @see genesis_markup $args Array.
		 */
		$open = apply_filters( "genesis_markup_{$args['context']}_open", $open, $args );

		/**
		 * Contextual filter to modify 'close' markup.
		 *
		 * @since 2.4.0
		 *
		 * @param string $close HTML tag being processed by the API.
		 * @param array  $args  Array with markup arguments.
		 *
		 * @see genesis_markup $args Array.
		 */
		$close = apply_filters( "genesis_markup_{$args['context']}_close", $args['close'], $args );

	} else {

		$open = $args['open'];
		$close = $args['close'];

	}

	if ( $open || $args['content'] ) {
		/**
		 * Non-contextual filter to modify 'open' markup.
		 *
		 * @since 2.4.0
		 *
		 * @param string $open HTML tag being processed by the API.
		 * @param array  $args Array with markup arguments.
		 *
		 * @see genesis_markup $args Array.
		 */
		$open = apply_filters( 'genesis_markup_open', $open, $args );
	}

	if ( $close || $args['content'] ) {
		/**
		 * Non-contextual filter to modify 'close' markup.
		 *
		 * @since 2.4.0
		 *
		 * @param string $open HTML tag being processed by the API.
		 * @param array  $args Array with markup arguments.
		 *
		 * @see genesis_markup $args Array.
		 */
		$close = apply_filters( 'genesis_markup_close', $close, $args );
	}

	if ( $args['echo'] ) {
		echo $open . $args['content'] . $close;

		return null;
	} else {
		return $open . $args['content'] . $close;
	}

}

add_action( 'after_setup_theme', 'genesis_xhtml_check' );
/**
 * Conditionally load XHTML markup.
 *
 * @since 2.4.0
 */
function genesis_xhtml_check() {

	if ( ! genesis_html5() ) {
		require_once( PARENT_DIR . '/lib/structure/xhtml.php' );

		_genesis_builtin_sidebar_params();
	}

}

/**
 * Merge array of attributes with defaults, and apply contextual filter on array.
 *
 * The contextual filter is of the form `genesis_attr_{context}`.
 *
 * @since 2.0.0
 *
 * @param string $context    The context, to build filter name.
 * @param array  $attributes Optional. Extra attributes to merge with defaults.
 * @param array  $args       Optional. Custom data to pass to filter.
 * @return array Merged and filtered attributes.
 */
function genesis_parse_attr( $context, $attributes = array(), $args = array() ) {

    $defaults = array(
        'class' => sanitize_html_class( $context ),
    );

    $attributes = wp_parse_args( $attributes, $defaults );

    // Contextual filter.
    return apply_filters( "genesis_attr_{$context}", $attributes, $context, $args );

}

/**
 * Build list of attributes into a string and apply contextual filter on string.
 *
 * The contextual filter is of the form `genesis_attr_{context}_output`.
 *
 * @since 2.0.0
 *
 * @param string $context    The context, to build filter name.
 * @param array  $attributes Optional. Extra attributes to merge with defaults.
 * @param array  $args       Optional. Custom data to pass to filter.
 * @return string String of HTML attributes and values.
 */
function genesis_attr( $context, $attributes = array(), $args = array() ) {

	$attributes = genesis_parse_attr( $context, $attributes, $args );

	$output = '';

	// Cycle through attributes, build tag attribute string.
	foreach ( $attributes as $key => $value ) {

		if ( ! $value ) {
			continue;
		}

		if ( true === $value ) {
			$output .= esc_html( $key ) . ' ';
		} else {
			$output .= sprintf( '%s="%s" ', esc_html( $key ), esc_attr( $value ) );
		}

	}

	$output = apply_filters( "genesis_attr_{$context}_output", $output, $attributes, $context, $args );

	return trim( $output );

}

/**
 * Helper function for use as a filter for when you want to prevent a class from being automatically
 * generated and output on an element that is passed through the markup API.
 *
 * @since 2.2.1
 *
 * @param array $attributes Existing attributes.
 * @return array Attributes with `class` set to empty string.
 */
function genesis_attributes_empty_class( $attributes ) {

	$attributes['class'] = '';

	return $attributes;

}

/**
 * Helper function for use as a filter for when you want to add screen-reader-text class to an element.
 *
 * @since 2.2.1
 *
 * @param array $attributes Existing attributes.
 * @return array Attributes with `screen-reader-text` added to classes
 */
function genesis_attributes_screen_reader_class( $attributes ) {

	$attributes['class'] .= ' screen-reader-text';

	return $attributes;

}

add_filter( 'genesis_attr_head', 'genesis_attributes_head' );
/**
 * Add attributes for head element.
 *
 * @since 2.2.0
 *
 * @param array $attributes Existing attributes for `head` element.
 * @return array Amended attributes for `head` element.
 */
 function genesis_attributes_head( $attributes ) {

 	$attributes['class'] = '';

	if ( ! is_front_page() ) {
		return $attributes;
	}

	$attributes['itemscope'] = true;
	$attributes['itemtype']  = 'https://schema.org/WebSite';

	return $attributes;

}

add_filter( 'genesis_attr_body', 'genesis_attributes_body' );
/**
 * Add attributes for body element.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for `body` element.
 * @return array Amended attributes for `body` element.
 */
function genesis_attributes_body( $attributes ) {

	$attributes['class']     = implode( ' ', get_body_class() );
	$attributes['itemscope'] = true;
	$attributes['itemtype']  = 'https://schema.org/WebPage';

	// Search results pages.
	if ( is_search() ) {
		$attributes['itemtype'] = 'https://schema.org/SearchResultsPage';
	}

	return $attributes;

}

add_filter( 'genesis_attr_site-header', 'genesis_attributes_header' );
/**
 * Add attributes for site header element.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for site header element.
 * @return array Amended attributes for site header element.
 */
function genesis_attributes_header( $attributes ) {

	$attributes['itemscope'] = true;
	$attributes['itemtype']  = 'https://schema.org/WPHeader';

	return $attributes;

}

add_filter( 'genesis_attr_site-title', 'genesis_attributes_site_title' );
/**
 * Add attributes for site title element.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for site title element.
 * @return array Amended attributes for site title element.
 */
function genesis_attributes_site_title( $attributes ) {

	$attributes['itemprop'] = 'headline';

	return $attributes;

}

add_filter( 'genesis_attr_site-description', 'genesis_attributes_site_description' );
/**
 * Add attributes for site description element.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for site description element.
 * @return array Amended attributes for site description element.
 */
function genesis_attributes_site_description( $attributes ) {

	$attributes['itemprop'] = 'description';

	return $attributes;

}

add_filter( 'genesis_attr_header-widget-area', 'genesis_attributes_header_widget_area' );
/**
 * Add attributes for header widget area element.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for header widget area element.
 * @return array Amended attributes for header widget area element.
 */
function genesis_attributes_header_widget_area( $attributes ) {

	$attributes['class'] = 'widget-area header-widget-area';

	return $attributes;

}

add_filter( 'genesis_attr_breadcrumb', 'genesis_attributes_breadcrumb' );
/**
 * Add attributes for breadcrumbs wrapper.
 *
 * @since 2.2.0
 *
 * @param array $attributes Existing attributes for breadcrumbs wrapper element.
 * @return array Amended attributes for breadcrumbs wrapper element.
 */
function genesis_attributes_breadcrumb( $attributes ) {

	$attributes['itemprop']  = 'breadcrumb';
	$attributes['itemscope'] = true;
	$attributes['itemtype']  = 'https://schema.org/BreadcrumbList';

	// Breadcrumb itemprop not valid on blog.
	if ( is_singular( 'post' ) || is_archive() || is_home() || is_page_template( 'page_blog.php' ) ) {
		unset( $attributes['itemprop'] );
	}

	return $attributes;

}

add_filter( 'genesis_attr_breadcrumb-link-wrap', 'genesis_attributes_breadcrumb_link_wrap' );
/**
 * Add attributes for breadcrumb item element.
 *
 * @since 2.2.0
 *
 * @param array $attributes Existing attributes for breadcrumb item element.
 * @return array Amended attributes for breadcrumb item element.
 */
function genesis_attributes_breadcrumb_link_wrap( $attributes ) {

	$attributes['itemprop']  = 'itemListElement';
	$attributes['itemscope'] = true;
	$attributes['itemtype']  = 'https://schema.org/ListItem';

	return $attributes;

}

add_filter( 'genesis_attr_search-form', 'genesis_attributes_search_form' );
/**
 * Add attributes for search form.
 *
 * @since 2.2.0
 *
 * @param array $attributes Existing attributes for search form element.
 * @return array Amended attributes for search form element.
 */
function genesis_attributes_search_form( $attributes ) {

	$attributes['itemprop']  = 'potentialAction';
	$attributes['itemscope'] = true;
	$attributes['itemtype']  = 'https://schema.org/SearchAction';
	$attributes['method']    = 'get';
	$attributes['action']    = home_url( '/' );
	$attributes['role']      = 'search';

	return $attributes;

}

add_filter( 'genesis_attr_nav-primary', 'genesis_attributes_nav' );
add_filter( 'genesis_attr_nav-secondary', 'genesis_attributes_nav' );
add_filter( 'genesis_attr_nav-header', 'genesis_attributes_nav' );
/**
 * Add typical attributes for navigation elements.
 *
 * Used for primary navigation, secondary navigation, and custom menu widgets in the header right widget area.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for navigation elements.
 * @return array Amended attributes for navigation elements.
 */
function genesis_attributes_nav( $attributes ) {

	$attributes['itemscope'] = true;
	$attributes['itemtype']  = 'https://schema.org/SiteNavigationElement';

	return $attributes;

}

add_filter( 'genesis_attr_nav-link-wrap', 'genesis_attributes_nav_link_wrap' );
/**
 * Add attributes for the span wrap around navigation item links.
 *
 * @since 2.2.0
 *
 * @param array $attributes Existing attributes for span wrap around navigation item links.
 * @return array Amended attributes for span wrap around navigation item links.
 */
function genesis_attributes_nav_link_wrap( $attributes ) {

	$attributes['class'] = '';
	$attributes['itemprop'] = 'name';

	return $attributes;

}

add_filter( 'genesis_attr_nav-link', 'genesis_attributes_nav_link' );
/**
 * Add attributes for the navigation item links.
 *
 * Since we're utilizing a filter that plugins might also want to filter, don't overwrite class here.
 *
 * @since 2.2.0
 *
 * @link https://github.com/copyblogger/genesis/issues/1226
 *
 * @param array $attributes Existing attributes for navigation item links.
 * @return array Amended attributes for navigation item links.
 */
function genesis_attributes_nav_link( $attributes ) {

	$class = str_replace( 'nav-link', '', $attributes['class'] );

	$attributes['class']    = $class;
	$attributes['itemprop'] = 'url';

	return $attributes;

}

add_filter( 'genesis_attr_structural-wrap', 'genesis_attributes_structural_wrap' );
/**
 * Add attributes for structural wrap elements.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for structural wrap elements.
 * @return array Amended attributes for structural wrap elements.
 */
function genesis_attributes_structural_wrap( $attributes ) {

	$attributes['class'] = 'wrap';

	return $attributes;

}

add_filter( 'genesis_attr_content', 'genesis_attributes_content' );
/**
 * Add attributes for main content element.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for `main` element.
 * @return array Attributes for `main` element.
 */
function genesis_attributes_content( $attributes ) {

	return $attributes;

}

add_filter( 'genesis_attr_taxonomy-archive-description', 'genesis_attributes_taxonomy_archive_description' );
/**
 * Add attributes for taxonomy archive description element.
 *
 * @since 2.2.1
 *
 * @param array $attributes Existing attributes for taxonomy archive description element.
 * @return array Amended attributes for taxonomy archive description element.
 */
function genesis_attributes_taxonomy_archive_description( $attributes ) {

	$attributes['class'] = 'archive-description taxonomy-archive-description taxonomy-description';

	return $attributes;

}

add_filter( 'genesis_attr_author-archive-description', 'genesis_attributes_author_archive_description' );
/**
 * Add attributes for author archive description element.
 *
 * @since 2.2.1
 *
 * @param array $attributes Existing attributes for author archive description element.
 * @return array Amended attributes for author archive description element.
 */
function genesis_attributes_author_archive_description( $attributes ) {

	$attributes['class'] = 'archive-description author-archive-description author-description';

	return $attributes;

}

add_filter( 'genesis_attr_cpt-archive-description', 'genesis_attributes_cpt_archive_description' );
/**
 * Add attributes for CPT archive description element.
 *
 * @since 2.2.1
 *
 * @param array $attributes Existing attributes for CPT archive description element.
 * @return array Amended attributes for CPT archive description element.
 */
function genesis_attributes_cpt_archive_description( $attributes ) {

	$attributes['class'] = 'archive-description cpt-archive-description';

	return $attributes;

}

add_filter( 'genesis_attr_date-archive-description', 'genesis_attributes_date_archive_description' );
/**
 * Add attributes for date archive description element.
 *
 * @since 2.2.1
 *
 * @param array $attributes Existing attributes for date archive description element.
 * @return array Amended attributes for date archive description element.
 */
function genesis_attributes_date_archive_description( $attributes ) {

	$attributes['class'] = 'archive-description date-archive-description archive-date';

	return $attributes;

}

add_filter( 'genesis_attr_blog-template-description', 'genesis_attributes_blog_template_description' );
/**
 * Add attributes for blog template description element.
 *
 * @since 2.2.1
 *
 * @param array $attributes Existing attributes for blog template description element.
 * @return array Amended attributes for blog template description element.
 */
function genesis_attributes_blog_template_description( $attributes ) {

	$attributes['class'] = 'archive-description blog-template-description';

	return $attributes;

}

add_filter( 'genesis_attr_posts-page-description', 'genesis_attributes_posts_page_description' );
/**
 * Add attributes for posts page description element.
 *
 * @since 2.2.1
 *
 * @param array $attributes Existing attributes for posts page description element.
 * @return array Amended attributes for posts page description element.
 */
function genesis_attributes_posts_page_description( $attributes ) {

	$attributes['class'] = 'archive-description posts-page-description';

	return $attributes;

}

add_filter( 'genesis_attr_entry', 'genesis_attributes_entry' );
/**
 * Add attributes for entry element.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for entry element.
 * @return array Amended attributes for entry element.
 */
function genesis_attributes_entry( $attributes ) {

	$attributes['class'] = implode( ' ', get_post_class() );

	if ( ! is_main_query() && ! genesis_is_blog_template() ) {
		return $attributes;
	}

	$attributes['itemscope'] = true;
	$attributes['itemtype']  = 'https://schema.org/CreativeWork';

	return $attributes;

}

add_filter( 'genesis_attr_entry-image', 'genesis_attributes_entry_image' );
/**
 * Add attributes for entry image element.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for entry image element.
 * @return array Amended attributes for entry image element.
 */
function genesis_attributes_entry_image( $attributes ) {

	$attributes['class']    = genesis_get_option( 'image_alignment' ) . ' post-image entry-image';
	$attributes['itemprop'] = 'image';

	return $attributes;

}

add_filter( 'genesis_attr_entry-image-link', 'genesis_attributes_entry_image_link' );
/**
 * Add attributes for entry image link element.
 *
 * @since 2.3.0
 *
 * @param array $attributes Existing attributes for entry image link element.
 * @return array Amended attributes for entry image link element.
 */
function genesis_attributes_entry_image_link( $attributes ) {

	$attributes['href']        = get_permalink();
	$attributes['aria-hidden'] = 'true';
	$attributes['class']       = 'entry-image-link';

	return $attributes;

}

add_filter( 'genesis_attr_entry-image-widget', 'genesis_attributes_entry_image_widget' );
/**
 * Add attributes for entry image element shown in a widget.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for entry image element shown in a widget.
 * @return array Amended attributes for entry image element shown in a widget.
 */
function genesis_attributes_entry_image_widget( $attributes ) {

	$attributes['class']    = 'entry-image attachment-' . get_post_type();
	$attributes['itemprop'] = 'image';

	return $attributes;

}

add_filter( 'genesis_attr_entry-image-grid-loop', 'genesis_attributes_entry_image_grid_loop' );
/**
 * Add attributes for entry image element shown in a grid loop.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for entry image element shown in a grid loop.
 * @return array Amended attributes for entry image element shown in a grid loop.
 */
function genesis_attributes_entry_image_grid_loop( $attributes ) {

	$attributes['itemprop'] = 'image';

	return $attributes;

}

add_filter( 'genesis_attr_entry-author', 'genesis_attributes_entry_author' );
/**
 * Add attributes for author element for an entry.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for author element for an entry.
 * @return array Amended attributes for author element for an entry.
 */
function genesis_attributes_entry_author( $attributes ) {

	$attributes['itemprop']  = 'author';
	$attributes['itemscope'] = true;
	$attributes['itemtype']  = 'https://schema.org/Person';

	return $attributes;

}

add_filter( 'genesis_attr_entry-author-link', 'genesis_attributes_entry_author_link' );
/**
 * Add attributes for entry author link element.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for entry author link element.
 * @return array Amended attributes for entry author link element.
 */
function genesis_attributes_entry_author_link( $attributes ) {

	$attributes['itemprop'] = 'url';
	$attributes['rel']      = 'author';

	return $attributes;

}

add_filter( 'genesis_attr_entry-author-name', 'genesis_attributes_entry_author_name' );
/**
 * Add attributes for entry author name element.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for entry author name element.
 * @return array Amended attributes for entry author name element.
 */
function genesis_attributes_entry_author_name( $attributes ) {

	$attributes['itemprop'] = 'name';

	return $attributes;

}

add_filter( 'genesis_attr_entry-time', 'genesis_attributes_entry_time' );
/**
 * Add attributes for time element for an entry.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for time element for an entry.
 * @return array Amended attributes for time element for an entry.
 */
function genesis_attributes_entry_time( $attributes ) {

	$attributes['itemprop'] = 'datePublished';
	$attributes['datetime'] = get_the_time( 'c' );

	return $attributes;

}

add_filter( 'genesis_attr_entry-modified-time', 'genesis_attributes_entry_modified_time' );
/**
 * Add attributes for modified time element for an entry.
 *
 * @since 2.1.0
 *
 * @param array $attributes Existing attributes for modified time element for an entry.
 * @return array Amended attributes for modified time element for an entry.
 */
function genesis_attributes_entry_modified_time( $attributes ) {

	$attributes['itemprop'] = 'dateModified';
	$attributes['datetime'] = get_the_modified_time( 'c' );

	return $attributes;

}

add_filter( 'genesis_attr_entry-title', 'genesis_attributes_entry_title' );
/**
 * Add attributes for entry title element.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for entry title element.
 * @return array Amended attributes for entry title element.
 */
function genesis_attributes_entry_title( $attributes ) {

	$attributes['itemprop'] = 'headline';

	return $attributes;

}

add_filter( 'genesis_attr_entry-content', 'genesis_attributes_entry_content' );
/**
 * Add attributes for entry content element.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for entry content element.
 * @return array Amended attributes for entry content element.
 */
function genesis_attributes_entry_content( $attributes ) {

	if ( ! is_main_query() && ! genesis_is_blog_template() ) {
		return $attributes;
	}

	$attributes['itemprop'] = 'text';

	return $attributes;

}

add_filter( 'genesis_attr_entry-meta-before-content', 'genesis_attributes_entry_meta' );
add_filter( 'genesis_attr_entry-meta-after-content', 'genesis_attributes_entry_meta' );
/**
 * Add attributes for entry meta elements.
 *
 * @since 2.1.0
 *
 * @param array $attributes Existing attributes for entry meta elements.
 * @return array Amended attributes for entry meta elements.
 */
function genesis_attributes_entry_meta( $attributes ) {

	$attributes['class'] = 'entry-meta';

	return $attributes;

}

add_filter( 'genesis_attr_archive-pagination', 'genesis_attributes_pagination' );
add_filter( 'genesis_attr_entry-pagination', 'genesis_attributes_pagination' );
add_filter( 'genesis_attr_adjacent-entry-pagination', 'genesis_attributes_pagination' );
add_filter( 'genesis_attr_comments-pagination', 'genesis_attributes_pagination' );
/**
 * Add attributes for pagination element.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for pagination element.
 * @return array Amended attributes for pagination element.
 */
function genesis_attributes_pagination( $attributes ) {

	$attributes['class'] .= ' pagination';

	return $attributes;

}

add_filter( 'genesis_attr_entry-comments', 'genesis_attributes_entry_comments' );
/**
 * Add attributes for entry comments element.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for entry comments element.
 * @return array Amended attributes for entry comments element.
 */
function genesis_attributes_entry_comments( $attributes ) {

	$attributes['id'] = 'comments';

	return $attributes;

}

add_filter( 'genesis_attr_comment', 'genesis_attributes_comment' );
/**
 * Add attributes for single comment element.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for single comment element.
 * @return array Amended attributes for single comment element.
 */
function genesis_attributes_comment( $attributes ) {

	$attributes['class']     = '';
	$attributes['itemprop']  = 'comment';
	$attributes['itemscope'] = true;
	$attributes['itemtype']  = 'https://schema.org/Comment';

	return $attributes;

}

add_filter( 'genesis_attr_comment-author', 'genesis_attributes_comment_author' );
/**
 * Add attributes for comment author element.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for comment author element.
 * @return array Amended attributes for comment author element.
 */
function genesis_attributes_comment_author( $attributes ) {

	$attributes['itemprop']  = 'author';
	$attributes['itemscope'] = true;
	$attributes['itemtype']  = 'https://schema.org/Person';

	return $attributes;

}

add_filter( 'genesis_attr_comment-author-link', 'genesis_attributes_comment_author_link' );
/**
 * Add attributes for comment author link element.
 *
 * @since 2.1.0
 *
 * @param array $attributes Existing attributes for comment author link element.
 * @return array Amended attributes for comment author link element.
 */
function genesis_attributes_comment_author_link( $attributes ) {

	$attributes['rel']      = 'external nofollow';
	$attributes['itemprop'] = 'url';

	return $attributes;

}

add_filter( 'genesis_attr_comment-time', 'genesis_attributes_comment_time' );
/**
 * Add attributes for comment time element.
 *
 * @since 2.1.0
 *
 * @param array $attributes Existing attributes for comment time element.
 * @return array Amended attributes for comment time element.
 */
function genesis_attributes_comment_time( $attributes ) {

	$attributes['datetime'] = esc_attr( get_comment_time( 'c' ) );
	$attributes['itemprop'] = 'datePublished';

	return $attributes;

}

add_filter( 'genesis_attr_comment-time-link', 'genesis_attributes_comment_time_link' );
/**
 * Add attributes for comment time link element.
 *
 * @since 2.1.0
 *
 * @param array $attributes Existing attributes for comment time link element.
 * @return array Amended attributes for comment time link element.
 */
function genesis_attributes_comment_time_link( $attributes ) {

	$attributes['itemprop'] = 'url';

	return $attributes;

}

add_filter( 'genesis_attr_comment-content', 'genesis_attributes_comment_content' );
/**
 * Add attributes for comment content container.
 *
 * @since 2.1.0
 *
 * @param array $attributes Existing attributes for comment content container.
 * @return array Amended attributes for comment content container.
 */
function genesis_attributes_comment_content( $attributes ) {

	$attributes['itemprop'] = 'text';

	return $attributes;

}

add_filter( 'genesis_attr_author-box', 'genesis_attributes_author_box' );
/**
 * Add attributes for author box element.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for author box element.
 * @return array Amended attributes for author box element.
 */
function genesis_attributes_author_box( $attributes ) {

	$attributes['itemprop']  = 'author';
	$attributes['itemscope'] = true;
	$attributes['itemtype']  = 'https://schema.org/Person';

	return $attributes;

}

add_filter( 'genesis_attr_sidebar-primary', 'genesis_attributes_sidebar_primary' );
/**
 * Add attributes for primary sidebar element.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for primary sidebar element.
 * @return array Amended attributes for primary sidebar element.
 */
function genesis_attributes_sidebar_primary( $attributes ) {

	$attributes['class']     = 'sidebar sidebar-primary widget-area';
	$attributes['role']      = 'complementary';
	$attributes['aria-label']  = __( 'Primary Sidebar', 'genesis' );
	$attributes['itemscope'] = true;
	$attributes['itemtype']  = 'https://schema.org/WPSideBar';

	return $attributes;

}

add_filter( 'genesis_attr_sidebar-secondary', 'genesis_attributes_sidebar_secondary' );
/**
 * Add attributes for secondary sidebar element.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for secondary sidebar element.
 * @return array Amended attributes for secondary sidebar element.
 */
function genesis_attributes_sidebar_secondary( $attributes ) {

	$attributes['class']     = 'sidebar sidebar-secondary widget-area';
	$attributes['role']      = 'complementary';
	$attributes['aria-label']  = __( 'Secondary Sidebar', 'genesis' );
	$attributes['itemscope'] = true;
	$attributes['itemtype']  = 'https://schema.org/WPSideBar';

	return $attributes;

}

add_filter( 'genesis_attr_site-footer', 'genesis_attributes_site_footer' );
/**
 * Add attributes for site footer element.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for site footer element.
 * @return array Amended attributes for site footer element.
 */
function genesis_attributes_site_footer( $attributes ) {

	$attributes['itemscope'] = true;
	$attributes['itemtype']  = 'https://schema.org/WPFooter';

	return $attributes;

}

add_filter( 'genesis_attr_footer-widget-area', 'genesis_attributes_footer_widget_area', 10, 3 );
/**
 * Add attributes for footer widget area wrapper elements.
 *
 * @since 2.5.0
 *
 * @param array  $attributes Existing attributes for footer widget area wrapper elements.
 * @param string $context    Not used. Markup context (ie. `footer-widget-area`).
 * @param array  $args       Markup arguments.
 * @return array Amended attributes for footer widget area wrapper elements.
 */
function genesis_attributes_footer_widget_area( $attributes, $context, $args ) {

	$column = ! empty( $args['params'] ) && ! empty( $args['params']['column'] ) ? $args['params']['column'] : 0;
	$attributes['class'] = sprintf( 'widget-area footer-widgets-%d ', $column ) . $attributes['class'];

	return $attributes;

}

/**
 * Add ID markup to the elements to jump to.
 *
 * @since 2.2.0
 *
 * @link https://gist.github.com/salcode/7164690
 */
function genesis_skiplinks_markup() {

	add_filter( 'genesis_attr_nav-primary', 'genesis_skiplinks_attr_nav_primary' );
	add_filter( 'genesis_attr_content', 'genesis_skiplinks_attr_content' );
	add_filter( 'genesis_attr_sidebar-primary', 'genesis_skiplinks_attr_sidebar_primary' );
	add_filter( 'genesis_attr_sidebar-secondary', 'genesis_skiplinks_attr_sidebar_secondary' );
	add_filter( 'genesis_attr_footer-widgets', 'genesis_skiplinks_attr_footer_widgets' );

}

/**
 * Add ID markup to primary navigation.
 *
 * @since 2.2.0
 *
 * @param array $attributes Existing attributes for primary navigation element.
 * @return array Amended attributes for primary navigation element.
 */
function genesis_skiplinks_attr_nav_primary( $attributes ) {

	$attributes['id']         = 'genesis-nav-primary';
	$attributes['aria-label'] = esc_html__( 'Main navigation', 'genesis' );

	return $attributes;

}

/**
 * Add ID markup to main content area.
 *
 * @since 2.2.0
 *
 * @param array $attributes Existing attributes for `main` element.
 * @return array Amended attributes for `main` element.
 */
function genesis_skiplinks_attr_content( $attributes ) {

	$attributes['id'] = 'genesis-content';

	return $attributes;

}

/**
 * Add ID markup to primary sidebar.
 *
 * @since 2.2.0
 *
 * @param array $attributes Existing attributes for primary sidebar element.
 * @return array Amended attributes for primary sidebar element.
 */
function genesis_skiplinks_attr_sidebar_primary( $attributes ) {

	$attributes['id'] = 'genesis-sidebar-primary';

	return $attributes;

}

/**
 * Add ID markup to secondary sidebar.
 *
 * @since 2.2.0
 *
 * @param array $attributes Existing attributes for secondary sidebar element.
 * @return array Amended attributes for secondary sidebar element.
 */
function genesis_skiplinks_attr_sidebar_secondary( $attributes ) {

	$attributes['id'] = 'genesis-sidebar-secondary';

	return $attributes;

}

/**
 * Add ID markup to footer widget area.
 *
 * @since 2.2.0
 *
 * @param array $attributes Existing attributes for footer widget area element.
 * @return array Amended attributes for footer widget area element.
 */
function genesis_skiplinks_attr_footer_widgets( $attributes ) {

	$attributes['id'] = 'genesis-footer-widgets';

	return $attributes;

}
